/* 
 *      XAB 1.0
 *
 *		Adam Beguelin
 *      Carnegie Mellon University
 *      Pittsburgh Supercomputing Center
 *      26 Nov 1991
 *
 *      xab.c
 *		
 *		Display xab events generated by a PVM program
 *
 * This program displays the events generated by a PVM program.
 * The events can be displayed graphically or simply textually.
 * The events are displayed at run time or may be saved to a log 
 * file which can be displayed postmortem.
 *
 * Revision 1.1 1991/12/01 adamb
 * Added getnint sanity checking
 */

/*
XXX
	+ add leftBitmaps for all the events...
	o scale time to porportional realtime
	+ make clock look better, have hours, min, sec  fields
	o make clock times monotonically increasing....
	+ add slider for speed of playback

*/

/*
  Each xab event must contain the following information:
	Proccess name and pid number of the process that generated
	the event and the event id.  ie, XAB_initsend is the event
	id for an initsend.
*/


#include <stdio.h>
#include <sys/param.h>

/*
 * Include files required for all Toolkit programs
 */

#include <X11/Intrinsic.h>  /* Intrinsics Definitions */
#include <X11/StringDefs.h> /* Standard Name-String definitions */

/*
 * Public include file for widgets we actually use in this file.
 */


#include <X11/Xaw/SmeBSB.h>             /* Athena bitmap for X11R4 */

#include <X11/Xaw/Label.h>              /* Athena Label Widget */
/* 
 *	Local include files
 */
#include "both.h"
#include "ab.h"
#include "mlist.h"
#include "xabmon.h"
#include "bstates.h"
#include "xsizes.h"

/*	External functions and vars */

char *xab_sstring();

extern mlist *mlistFindSer();
extern XtAppContext app_context;
extern Widget machBox;
extern XtInputId getFileId;
extern int getFileActive;
extern int buttonState;
extern Pixmap sndPixmap, snddPixmap, rcvPixmap, rcvdPixmap, clockPixmap;
extern Pixmap barPixmap, bardPixmap, probePixmap, putPixmap, getPixmap;
extern Widget clockLabel;
extern void wakeUpInput();
extern int delayMs;



/* internal functions */
extern ups* upsFindCompPid();
extern void printUps();

/*	Global vars used elsewhere too */

XtIntervalId ioTimer;
int ioTimerActive = 0;
int mypid;
int printit = 0;
int abfid;
FILE *abin;

/*	Global variable for this file */

ups *ups_head = (ups *)0;
char tmpstring[200];
Pixmap monitorPixmap;

void
ab_getput(curevent, mcomp, mpid, rc, serial, count)
int curevent, mpid,rc,serial,count;
char *mcomp;
{}


/*	msgTypeCheck
*	check this message between these two processes for sanity
*/

void
msgTypeCheck(serial, sps, dps)
int serial;	/* serial number of the message */
ups *sps,	/* source process */
	*dps;	/* dest process */
{
	mlist *sml, *dml;
	mdesc *smdesc, *dmdesc, *t;
	int pc, gc;

	sml = mlistFindSer(sps->snd, serial);
	dml = mlistFindSer(dps->rcv, serial);

	/* if one of the messages doesn't exist then return */
	if ((!sml) || (!dml))
		return;

	/* check the types from the send message to the dest */
	for (smdesc = sml->head.mdesc_next,
		 dmdesc = dml->head.mdesc_next;
		 	smdesc && dmdesc; 
		 		smdesc = smdesc->mdesc_next,
		 		dmdesc = dmdesc->mdesc_next) {

		/* make sure these types match */
		if (dmdesc->type != smdesc->type) {
			printf("1Warning! (%s, %d) -> (%s, %d) %d get's don't match puts!\n",
				sps->pname, sps->pid, dps->pname, dps->pid, serial);
			/*
			printUps(ups_head);
			*/
			return;
			}

		/* if the numbers match then iterate */
		if (smdesc->number == dmdesc->number ) 
			continue;

		/* count all the puts of this type, advancing smdesc */
		for(pc = smdesc->number, t = smdesc; t->mdesc_next; t = t->mdesc_next) {
			if (t->type != t->mdesc_next->type) break;
			pc += t->mdesc_next->number;
			}
		smdesc = t;

		/* count all the gets of this type, advancing dmdesc */
		for(gc = dmdesc->number, t = dmdesc; t->mdesc_next; t = t->mdesc_next) {
			if (t->type != t->mdesc_next->type) break;
			gc += t->number;
			}
		dmdesc = t;

		if (pc != gc) {
			printf("Warning! (%s, %d) -> (%s, %d) %d get's don't match puts!\n",
				sps->pname, sps->pid, dps->pname, dps->pid, serial);
			/*
			printUps(ups_head);
			*/
			return;
			}

		}

}
	
/*	msgSanity
*	check the ups list for the indicated messsage and make
*	sure things match up
*/

int
msgSanity(event, serial, scomp, spid, dcomp, dpid)
int		event,	/* XAB event, either XAB_snd or XAB_rcv_done */
		serial;	/* serial number of the message */
char 	*scomp;	/* source component name */
int 	spid;	/* source pid */
char 	*dcomp;	/* dest component name */
int		dpid;	/* dest pid */
{
	ups *sps, *dps;

	sps = upsFindCompPid(ups_head, scomp, spid);
	if (!sps)
		if (event == XAB_snd)
			bail("msgSanity: send from nonexistent process!");
		else
			/* inconclusive, rcvd from process that we don't know about yet */
			return(0);

	/* if this is not a broadcast */
	if (dpid != -1) {
		dps = upsFindCompPid(ups_head, dcomp, dpid);
		if (!dps)
			if (event == XAB_rcv_done)
				bail("msgSanity: rcv from nonexistent process!");
			else
				/* inconclusive, snd to process that we don't know about yet */
				return(0);
		msgTypeCheck(serial, sps, dps);
		}

	/* This is a broadcast, check all dps of this component  */
	
	for (dps = ups_head; dps != 0; dps = dps->ups_next) {
		/* if not the source but the dest comp */
		if ((dps->pid != spid) && (!strcmp(dps->pname,dcomp)))
			msgTypeCheck(serial, sps, dps);
		}
}

/*	printMdesc(hd)
*	print the message descriptors 
*/

void
printMdesc(hd)
mdesc *hd;
{
	printf("\t type\t number\t rc\n");
	while(hd) {
		printf("\t %s\t %d %d\n", deftostr(hd->type), hd->number, hd->rc);
		hd = hd->mdesc_next;
		}
}

/*	deleteMdesc
*	delete the message descriptor list
*/

void
deleteMdesc(hd)
mdesc *hd;
{
	mdesc *t;
	while(hd) {
		t = hd;
		hd = hd->mdesc_next;
		free(t);
		}
}

/*	printMlist
*	print the message list
*/

void
printMlist(hd)
mlist *hd;
{
	while(hd) {
		printf("%d (%s, %d), type %d len %d\n", 
			hd->serial, hd->comp, hd->pid, hd->mtype, hd->mlen);
		printMdesc(hd->head.mdesc_next);
		hd = hd->next;
		}
}

/*	deleteMlist
*	delete the message list
*/

void
deleteMlist(hd)
mlist *hd;
{
	mlist *t;
	while(hd) {
		t = hd;
		deleteMdesc(hd->head.mdesc_next);
		hd = hd->next;
		free(t);
		}
}

/*
*	printUps
*	print the list of user processes
*/

void
printUps(hd)
ups *hd;
{
	while(hd) {
		printf("\n========Process List============\n");
		printf("(%s, %d)\n", hd->pname, hd->pid);
		printf("\t host %s, arch %s\n", 
			((hd->host)?(hd->host):"????"),
			((hd->arch)?(hd->arch):"????"));
		printf("\t parent (%s, %d)\n", 
			((hd->parent_name)?(hd->parent_name):"????"),
			hd->parent_pid);
		if (hd->snd) {
			printf("Sends\n");
			printMlist(hd->snd);
			}
		if(hd->rcv) {
			printf("Receives\n");
			printMlist(hd->rcv);
			}
		hd = hd->ups_next;
		}
	printf("====================\n\n");
}


/*
*	deleteUpsCompPid
*	delete the entry for this component,pid from the ups list
*	return new ups head
*/

ups*
deleteUpsCompPid(uhead, comp, pid)
ups *uhead;
char *comp;
int pid;
{
	ups *t, *tmp;

	/* if the list is null then return an error */
	if (uhead == (ups *)0) {
		fprintf(stderr, "deleteUpsCompPid: no such component (%s, %d)\n", 
			comp, pid);
		return(uhead);
		}

	/* if the head is the element to remove, unlink it  */
	if ((uhead->pid ==  pid) && (strcmp(uhead->pname, comp) == 0))  {
		t = uhead;
		uhead = uhead->ups_next;
		}
	else {
		for (t = uhead; t->ups_next != (ups *)0; t = t->ups_next)  {
			/* if the next entry should be deleted, unlink it */
			if((t->ups_next->pid==pid) && (!strcmp(t->ups_next->pname,comp))){
				tmp = t->ups_next;
				t->ups_next = t->ups_next->ups_next;
				t = tmp;
				break;
				}
			}
		}

	/* if the entry wasn't found, return an error */
	if (t == (ups *)0) {
		fprintf(stderr, "deleteUpsCompPid: no such component (%s, %d)\n", 
			comp, pid);
		return(uhead);
		}

	/* free up t */
	deleteMlist(t->snd);
	deleteMlist(t->rcv);
	if(t->pname) 		free(t->pname);
	if(t->arch) 		free(t->arch);
	if(t->host) 		free(t->host);
	if(t->parent_name) 	free(t->parent_name);
	XtDestroyWidget(t->widget);

	free(t);

	return(uhead);
}

/*
*	upsFindCompPid
*	find the ups entry with the given name and pid
*/

ups*
upsFindCompPid(uhead,comp,pid)
ups *uhead;
char *comp;
int pid;
{
	ups *t;

	/* look through the list and return the matching pid */
	for (t = uhead; t != 0;  t = t->ups_next) {
		if ((t->pid == pid)  && (!strcmp(t->pname,comp))) return(t);
		}
	
	return((ups *)0);
}

/*	ab_rcv
*	add and entry for this message serial number
*/
void
ab_rcv_done(comp,pid)
char *comp;	/* name of the ps that did the snd */
int pid; 	/* pid of the ps that did the snd */
{
	ups *myUps;
	mlist *tmp;
	char scomp[MAXUPROCNAMELEN];

	/* | src_comp | src_pid | msg_type | length | serial | */

	myUps = upsFindCompPid(ups_head, comp, pid);

	if (! myUps ) { /* We've got problems, this should never happen */
		fprintf(stderr, "ab_rcv_done(%s, %d): no such component\n", comp, pid);
		fprintf(stderr, "Bailing\n");
		exit(-1);
		}

	/* add this message to the ups entry */
	tmp = tmalloc(1, mlist);
	if (! tmp) bail("ab_rcv_done: out of memory");

	/* link in the message */
	tmp->next = myUps->rcv;
	myUps->rcv = tmp;
	
	/* fill 'er up */
	/*
	if(getstring(scomp)) bail("ab_rcv_done: getting scomp");
	*/
	myUps->rcv->comp = xab_sstring(scomp);
	/*
	if(getnint(&myUps->rcv->pid, 1)) bail("ab_rcv_done: getting pid");		
	if(getnint(&myUps->rcv->mtype, 1)) bail("ab_rcv_done: getting type");	
	if(getnint(&myUps->rcv->mlen, 1)) bail("ab_rcv_done: getting mlen");
	if(getnint(&myUps->rcv->serial, 1)) bail("ab_rcv_done: getting type");	
	*/
	myUps->rcv->head.number		= 0;
	myUps->rcv->head.type 		= 0;
	myUps->rcv->head.rc 		= 0;
	myUps->rcv->head.mdesc_next = 0;

	/* the headers will be loaded by ab_get calls */
	
	/* check the message for sanity */
	/*
	msgSanity(XAB_rcv_done,  myUps->rcv->serial, 
		myUps->rcv->comp, myUps->rcv->pid, comp, pid);
	*/
}

/*	ab_snd
*	store the message headers in the ups structure.
*	Need to do the checking at some point against the gets...
*	These may come in out of order.  hmmm...
*/

void
ab_snd(comp, pid)
char *comp;	/* name of the ps that did the snd */
int pid; 	/* pid of the ps that did the snd */
{
	ups *myUps;
	int n, b[3];
	char dcomp[MAXUPROCNAMELEN];
	mlist *tmp;


	/* | dst_comp | dst_pid | msg_type | serial |
		 num hdrs | (type | num | rc )* | */

	myUps = upsFindCompPid(ups_head, comp, pid);

	if (! myUps ) { /* We've got problems, this should never happen */
		fprintf(stderr, "ab_snd(%s, %d): no such component\n", comp, pid);
		fprintf(stderr, "Bailing\n");
		exit(-1);
		}

	/* add this message to the ups entry */
	tmp = tmalloc(1, mlist);
	if (! tmp) bail("ab_snd: out of memory");
	
	/* link in the message */
	tmp->next = myUps->snd;
	myUps->snd = tmp;
	
	/* fill 'er up */
	/* get the component name */
	/*
	if(getstring(dcomp)) bail("ab_snd: getting dcomp");	
	*/
	myUps->snd->comp = xab_sstring(dcomp);
	/* this is the dest for the snd */
	/*
	if(getnint(&myUps->snd->pid, 1)) bail("ab_snd: getting pid");		
	if(getnint(&myUps->snd->mtype, 1)) bail("ab_snd: getting mtype");	
	myUps->snd->mlen = 0;
	if(getnint(&myUps->snd->serial, 1)) bail("ab_snd: getting serial");	
	*/
	myUps->snd->head.number		= 0;
	myUps->snd->head.type 		= 0;
	myUps->snd->head.rc 		= 0;
	myUps->snd->head.mdesc_next = 0;

	/* load the headers into the ups snd structure */
	/* get number of headers */
	/*
	if (getnint(&n, 1)) bail("ab_snd: no num"); 
	*/
	while (n--) {
		/* type, number, return code */
		/*
		if (getnint(b, 3)) bail("ab_snd: no type info"); 
		*/
		mdescAdd(&myUps->snd->head, b[0], b[1], b[2]);
		}
}

/* 	ab_get
*	handle get calls
*/
void
ab_get(event, comp, pid, rc)
int event;	/* type of get this is */
char *comp;	/* component that did the get */
int pid,	/* pid of the component that did the get */
	rc;		/* return code from the get */
{
	/* | serial | cnt  | */

	ups *myUps;
	int serial, cnt;
	mlist *m;

	
	myUps = upsFindCompPid(ups_head, comp, pid);

	if (! myUps ) { /* We've got problems, this should never happen */
		fprintf(stderr, "ab_get(%s, %d): no such component\n", comp, pid);
		fprintf(stderr, "Bailing\n");
		exit(-1);
		}

	/* find the message list with this serial number */
	/*
	getnint(&serial, 1);
	*/

	m = mlistFindSer(myUps->rcv, serial);
	if (! m) { /* What? No such serial number? */
		sprintf(tmpstring,"ab_get: %d no such serial number!\n", serial);
		bail(tmpstring);
		}

	/*
	getnint(&cnt, 1);
	*/
	mdescAdd(&m->head, event, cnt,  rc);

	/*
	msgSanity(XAB_rcv_done, serial, m->comp, m->pid, comp, pid);
	*/


}

/*	ab_leave
*	delete the user process from the ups list and
*	possibly reenroll xab
*/

void
ab_leave(comp, pid)
char *comp;
int pid;
{

	/* don't delete this since you want it to show the exited ps */
	/*
	ups_head = deleteUpsCompPid(ups_head,comp,pid);
	*/

	/* just get out will ya? */
	return;
}

/*
*	ab_showevent
*	add an event to the widget 
*/

void
ab_showevent(comp, pid, eventstr, evid)
char *comp;
int pid;
char *eventstr;
int evid;
{
	Pixmap tpm;
	ups *t;
	char *cp;

	t = upsFindCompPid(ups_head, comp, pid);


	switch(evid) {
	case XAB_snd:
	case XAB_vsnd:
		tpm = sndPixmap;
		break;
	case XAB_snd_done:
		tpm = snddPixmap;
		break;
	case XAB_vrcv_done:
	case XAB_rcv_done:
		tpm = rcvdPixmap;
		break;
	case XAB_rcv:
		tpm = rcvPixmap;
		break;
	case XAB_barrier:
		tpm = barPixmap;
		break;
	case XAB_barrier_done:
		tpm = bardPixmap;
		break;
	case XAB_putnint:
	case XAB_putnshort:
	case XAB_putnlong:
	case XAB_putnfloat:
	case XAB_putndfloat:
	case XAB_putncplx:
	case XAB_putbytes:
	case XAB_putstring:
		tpm = putPixmap;
		break;
	case XAB_getnint:
	case XAB_getnshort:
	case XAB_getnlong:
	case XAB_getnfloat:
	case XAB_getndfloat:
	case XAB_getncplx:
	case XAB_getbytes:
	case XAB_getstring:
		tpm = getPixmap;
		break;
	case XAB_probe:
		tpm = probePixmap;
		break;
	default:
		tpm = monitorPixmap;
		break;
	}

	if (t) {	/* add in the event string */
		/* shorten the host name */
/*		if (cp=xab_sstring(index(t->host,'.'))) *cp = '\0';*/
		if (cp=index(t->host,'.')) *cp = '\0';
		sprintf(tmpstring,"(%s, %d): %s %s: %s",
			t->pname, t->pid, t->host, t->arch, eventstr);
		XtVaSetValues(t->widget,
			XtNlabel, tmpstring,
			XtNjustify, XtJustifyLeft,
			XtNwidth, MACHENTRYWIDTH,
			XtNleftBitmap, tpm,
			NULL);
		}
	else {		/* component doesn't exist */	
		fprintf(stderr, "Component (%s,%d) event %s doesen't exist\n",
			comp, pid, eventstr);
		return;
		}


}

/*
*	ab_addComp
* 	add the component to the ups list.  It may already exist.
*	If it exisits then just set the host
*/

void 
ab_addComp(comp, pid, host)
char *comp;	/* the component name of the enrolling process */
int pid;	/* the pid of the enrolling process */
char *host;
{
	ups *t;

	t = upsFindCompPid(ups_head, comp, pid);

	if (t) {	/* add in the host name */
		free(t->host);
		t->host = xab_sstring(host);
		sprintf(tmpstring,"(%s, %d): %s %s",t->pname, t->pid, t->host, t->arch);
		XtVaSetValues(t->widget,
			XtNlabel, tmpstring,
			NULL);
		}
	else {		/* create a new ups entry */
		Arg arg[5];
		int n;
		/* create a new process entry in the user process list */
		t = tmalloc(1, ups);
		t->pname = xab_sstring(comp);
		t->arch  = (char *)0;	/* unknown for non-initiated processes */
		t->pid	 = pid;
		t->host  = xab_sstring(host);
		/* there is no parent for this guy */
		t->parent_name   = (char *) 0;
		t->parent_pid	 = -1;

		/* link it into the list */
		t->ups_next = ups_head;
		ups_head  	= t;

		/* null out the snd and rcv lists */
		t->snd = (mlist *)0;
		t->rcv = (mlist *)0;

		/* add a new widget for this process */
		sprintf(tmpstring,"(%s, %d): %s <>",comp,pid, host);
		n = 0;
		XtSetArg(arg[n], XtNwidth, MACHENTRYWIDTH); n++;
		XtSetArg(arg[n], XtNjustify, XtJustifyLeft); n++;
		XtSetArg(arg[n], XtNlabel, tmpstring); n++;
    	t->widget = XtCreateManagedWidget(
        	comp, 				/* widget name */
        	labelWidgetClass,
        	machBox,            /* parent widget*/
        	arg,               /* argument list */
        	n                   /* arg list size */
        	);

		}

}

/*
*	ab_initiate
*	create a new entry in the ups list
*/

void
ab_initiate(parent_comp, parent_pid)
char *parent_comp;	/* component that did the initialization */
int parent_pid;	/* pid of the component that did the initialization */
{
	Arg arg;
	int n;

	char *seps = ", \t\n";

	ups *t;
	int pid;
	char aout[MAXUPROCNAMELEN], arch[MAXUPROCNAMELEN];

    /* | new_ps_pid | aout | arch | */
	/*
	getnint(&pid, 1);
	getstring(aout);
	getstring(arch);
	*/

	/* create a new process entry in the user process list */
	t = tmalloc(1, ups);
	t->pname = xab_sstring(aout);
	t->arch  = xab_sstring(arch);
	t->pid	 = pid;
	t->host  = (char *)0;	/* host name for the process (derived at enroll)*/
	t->parent_name   = xab_sstring(parent_comp);
	t->parent_pid	 = parent_pid;

	/* link it into the list */
	t->ups_next = ups_head;
	ups_head  	= t;

	/* null out the snd and rcv lists */
	t->snd = (mlist *)0;
	t->rcv = (mlist *)0;

	/* add a new widget for this process */

   	t->widget = XtCreateManagedWidget(
       	aout,  				/* widget name */
       	labelWidgetClass,
       	machBox,            /* parent widget*/
       	NULL,               /* argument list */
       	0                   /* arg list size */
       	);

	/* add the string */
	sprintf(tmpstring, "(%s, %d): <> %s",t->pname, t->pid, t->arch);
	XtVaSetValues(t->widget,
		XtNlabel, tmpstring,
		XtNjustify, XtJustifyLeft,
		XtNwidth, MACHENTRYWIDTH,
		NULL);

}

/* get messages and do what needs to be done */

void 
ab_handle(line)
char *line;
{
	char *curtok, *sep = " ,\t\n";
	static long tsBase = -1, tuBase;
	long th, tm, ts, tu;
	char evname[20], comp[MAXUPROCNAMELEN], archhost[MAXUPROCNAMELEN];
	char evstr[100], tmpstring[20];
	char *tmpline;

	int mlen, mtype, mpid;
	char mcomp[MAXUPROCNAMELEN];
	int curevent, rc;

	int serial, type, len, instance, cnt;
	char component[MAXUPROCNAMELEN];

	tmpline = (char *)malloc(strlen(line)+1);
	strcpy(tmpline, line);
	/* get the time tokens */
	ts = atol(strtok(line, sep));
	tu = atol(strtok(0,    sep));

	/* save the base time */
	if (tsBase == -1) {
		tsBase = ts;
		tuBase = tu;
		/*
		ts = tu = 0;
		*/
		}

	if (tu >= tuBase) {
		tu = tu - tuBase;
		ts = ts - tsBase;
		}
	else {
		tu = tu + 1000000 - tuBase;
		ts = ts - tsBase - 1;
		}

	th = ts / 3600;
	tm = (ts - (th * 3600))/60;
	ts = ts - th * 3600 - tm * 60;

	sprintf(tmpstring, "  %02ld:%02ld:%02ld:%05ld", th, tm, ts, tu);
	XtVaSetValues(clockLabel,
		XtNlabel, tmpstring,
		XtNleftBitmap, clockPixmap,
		XtNjustify, XtJustifyLeft,
		XtNwidth, CLOCKWIDTH,
		XtNheight, 20,
		NULL);
	/* get the event token */
	curevent = atoi(strtok(0, sep));
	strcpy(evname, strtok(0, sep));

	/* get the component/pid */
	strcpy(mcomp, strtok(0, sep));
	mpid = atoi(strtok(0, sep));

	/* YYY missing mlen data now */
	
	switch(curevent) {
		case XAB_initsend:
			{
			int serial;

			strtok(0, sep);
			serial = atoi(strtok(0, sep));
			sprintf(evstr, "%s, Serial %d", evname, serial);
			ab_showevent(mcomp, mpid, evstr, curevent);
			}
			break;

		case XAB_putnint:
		case XAB_putnshort:
		case XAB_putnlong:
		case XAB_putnfloat:
		case XAB_putndfloat:
		case XAB_putncplx:
		case XAB_putbytes:
		case XAB_putstring:

		case XAB_getnint:
		case XAB_getnshort:
		case XAB_getnlong:
		case XAB_getnfloat:
		case XAB_getndfloat:
		case XAB_getncplx:
		case XAB_getbytes:
		case XAB_getstring:

			{ 
			int rc, serial, count;

			strtok(0, sep);
			rc = atoi(strtok(0, sep));
			strtok(0, sep);
			serial = atoi(strtok(0, sep));
			strtok(0, sep);
			count = atoi(strtok(0, sep));

			sprintf(evstr, "%s, Serial %d, Count %d, Rc %d", 
				evname, serial, count, rc);
			ab_showevent(mcomp, mpid, evstr, curevent);
			break;
			}

		case XAB_barrier:
			{
			char *bname;
			int bnum;

			strtok(0,sep);
			bname = xab_sstring(strtok(0, sep));
			strtok(0,sep);
			bnum = atoi(strtok(0,sep));
			sprintf(evstr, "%s, Name %s, Number %d",
				evname, bname, bnum);
			free(bname);
			ab_showevent(mcomp, mpid, evstr, curevent);
			}
			break;
		case XAB_barrier_done:
			ab_showevent(mcomp, mpid, evname, curevent);
			break;
		case XAB_enroll:
			{
			/*
			show the new process as enrolled, the process may already
			exist but be inactive from an initiate.
			*/

			if (printit) printf("enroll\n");
			/* | hostname | */
			ab_addComp(mcomp, mpid, strtok(0, sep));
			ab_showevent(mcomp, mpid, evname, curevent);
			}
			break;
		case XAB_initiate:
		case XAB_initiateM:

			/* 
			need to add a new process to the list, it will not
			be active until an enroll/whoami is done 
			*/

			strtok(0, sep);
			strcpy(comp,strtok(0, sep));
			strtok(0, sep);
			strcpy(archhost,strtok(0, sep));
			if (curevent == XAB_initiate)
				sprintf(evstr, "%s, Comp %s, Arch %s",
					evname, comp, archhost);
			else
				sprintf(evstr, "%s, Comp %s, Host %s",
					evname, comp, archhost);
			/*
			ab_initiate(mcomp, mpid);
			*/
			ab_showevent(mcomp, mpid, evstr, curevent);
			break;
		case XAB_initiate_done:
		case XAB_initiateM_done:
			strtok(0, sep);
			sprintf(evstr, "%s, Pid %d", evname, atoi(strtok(0,sep)));
			ab_showevent(mcomp, mpid, evstr, curevent);
			break;
		case XAB_leave:
			/*
			remove process from the active list
			*/

			ab_showevent(mcomp, mpid, evname, curevent);
			ab_leave(mcomp, mpid);
			break;
		case XAB_probe:
			ab_showevent(mcomp, mpid, evname, curevent);
			break;
		case XAB_pstatus:
			if (printit) printf("pstatus\n");
			ab_showevent(mcomp, mpid, evname, curevent);
			break;
		case XAB_rcv:
		case XAB_vrcv:
			if (printit) printf("rcv of type %d\n",type);
			strtok(0, sep);
			sprintf(evstr, "%s, Type %d", evname, atoi(strtok(0, sep)));
			ab_showevent(mcomp, mpid, evstr, curevent);
			break;
		case XAB_rcv_done:
		case XAB_vrcv_done:
			{
			int pid, type, len, ser;
			char *comp;

			strtok(0, sep);
			rc = atoi(strtok(0,sep));
			strtok(0, sep);
			comp = xab_sstring(strtok(0, sep));
			strtok(0, sep);
			pid = atoi(strtok(0, sep));
			strtok(0, sep);
			type = atoi(strtok(0, sep));
			strtok(0, sep);
			len = atoi(strtok(0, sep));
			strtok(0, sep);
			ser = atoi(strtok(0, sep));

			sprintf(evstr, "%s ser %d rc %d type %d len %d (%s,%d)",
				evname, ser, rc, type, len, comp, pid);

			if (printit) printf("rcv_done, rc %d\n", rc);
			/*
			getnint(&rc, 1);
			ab_rcv_done(mcomp, mpid);
			*/
			ab_showevent(mcomp, mpid, evstr, curevent);
			}
			break;
		case XAB_rcvmulti:
		case XAB_rcvinfo:
		case XAB_ready:
			break;
		/* XXX  is this really what to do for a snd? */
		case XAB_vsnd:
		case XAB_snd:
			{
			int rc, pid, type, len, ser;
			char *comp;
			/*
			ab_snd(mcomp, mpid);
			*/

			strtok(0, sep);
			rc = atoi(strtok(0, sep));
			strtok(0, sep);
			/* XXX never really use comp... */
			comp = xab_sstring(strtok(0, sep));
			strtok(0, sep);
			pid = atoi(strtok(0, sep));
			strtok(0, sep);
			type = atoi(strtok(0, sep));
			strtok(0, sep);
			len = atoi(strtok(0, sep));
			strtok(0, sep);
			ser = atoi(strtok(0, sep));
			sprintf(evstr, "%s, Type %d, Serial %d, Count %d, Rc %d", 
				evname, type, ser, len, rc);
			ab_showevent(mcomp, mpid, evstr, curevent);
			}
			break;
		case XAB_status:
			if (printit) printf("status\n");
			ab_showevent(mcomp, mpid, evname, curevent);
			break;
		case XAB_terminate:
			/*
			show the process was teminated and by whom
			*/
			if (printit) printf("terminate\n");
			ab_showevent(mcomp, mpid, evname, curevent);
			break;
		case XAB_waituntil:
			if (printit) printf("waituntil\n");
			ab_showevent(mcomp, mpid, evname, curevent);
			break;
		case XAB_whoami:
			if (printit) printf("whoami\n");
			ab_addComp(mcomp, mpid, strtok(0, sep));
			ab_showevent(mcomp, mpid, evname, curevent);
			break;
		
		default:
			printf("%d no such event\n", curevent);
			printf("line is:\n%s\n", tmpline);
			break;

		} /* switch(curevent) */


	free(tmpline);
	/*
	system("clear");
	printUps(ups_head);
	*/
}

/* 
* get_file_input
* this routine reads the input file and calls the appropriate
* functions to display the input
*/
int
get_file_input(client_data, fid, id)
XtPointer client_data;
int *fid;
XtInputId *id;
{
	/* the event is on two lines */
	char line[BUFSIZ];
	static char buf[BUFSIZ], *bp = buf;
	static int bs = BUFSIZ;
	int nbytes, i, nl;
	char *p;

	/* read in some bytes */
	if ((nbytes = read(*fid, bp, bs)) == -1) {
		perror("get_file_input");
		exit(-1);
		}
	
	/* new buffer size */
	bs = bs - nbytes;
	bp = bp + nbytes;

	/* count the number of new lines in the buffer */
	for (i = 0, nl = 0; i < BUFSIZ-bs; i++)
		if (buf[i] == '\n') nl++;

	/* if we have a full record (at least two new lines) */
	if (nl > 1) {
		/* copy the first two lines to line[] */
		for (i = 0, nl = 0; nl < 2; i++) {
			if ((line[i] = buf[i]) == '\n') nl++;
			}
		line[i] = '\0';

		/* move remaining bytes to beginning of buffer */
		bcopy(buf+i, buf, BUFSIZ-bs-i);

		/* set the bufsize and pointer to the right values */
		bs = bs + i;
		bp = buf + BUFSIZ - bs;

		/* now that we have a record we may need to stop looking for more */
		if (buttonState == FWDSTEP) {
			buttonState = STOPPED;
			if (getFileActive) {
				XtRemoveInput(getFileId);
				getFileActive = 0;
				}
			}

		if (buttonState == PLAYING && delayMs) {
			if (getFileActive) {
				XtRemoveInput(getFileId);
				getFileActive = 0;
				}
			if (! ioTimerActive) {
				ioTimer = XtAppAddTimeOut(app_context, delayMs, wakeUpInput, 0);
				ioTimerActive = 1;
				}
			}
			

		/* now line has the event (whew) so deal with it */
		ab_handle(line);
		}

}

void
main(argc, argv)
int argc;
char *argv[];
{

	/* call the X setup stuff */
	xsetup(&argc, argv);

	/* 
		Parse command line arguments 
	*/

	if (argc > 2) goto usage;

	/* open the output file */
	abin = stdin;
	if (argc == 2) {
		/* open the input file */
		abin = fopen(argv[1], "r");
		/* if fopen fails */
		if (!abin) {
			perror(argv[0]);
			exit(-1);
			}
		}

	XtAppMainLoop(app_context);

usage:
	fprintf(stderr,"usage: %s -d\n", argv[0]);
	exit(-1);
}
